<?php

declare(strict_types=1);

namespace BadamSoft\WooProductExporter\Access;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

use function __;
use function apply_filters;
use function get_current_user_id;
use function user_can;

class AccessManager {
    public const CAP_MANUAL_EXPORT     = 'prodexfo_access_manual_export';
    public const CAP_PREVIEW_EXPORT    = 'prodexfo_access_preview';
    public const CAP_TEMPLATES         = 'prodexfo_access_templates';
    public const CAP_HISTORY_VIEW      = 'prodexfo_access_history';
    public const CAP_HISTORY_ACTIONS   = 'prodexfo_access_history_actions';
    public const CAP_HISTORY_MANAGE    = 'prodexfo_access_history_manage';
    public const CAP_SETTINGS          = 'prodexfo_access_settings';

    /**
     * Note: Role-based access control (capability matrix / role manager) is a PRO-only feature.
     * The FREE plugin intentionally does not implement or apply a role permission matrix.
     */

    /**
     * @return array<string, array<string, string>>
     */
    public function get_capabilities(): array {
        $capabilities = [
            self::CAP_MANUAL_EXPORT     => [
                'label'       => __( 'Manual exports', 'badamsoft-product-exporter-for-woocommerce' ),
                'description' => __( 'Access the main Product CSV Export page and run manual exports.', 'badamsoft-product-exporter-for-woocommerce' ),
            ],
            self::CAP_PREVIEW_EXPORT    => [
                'label'       => __( 'Export preview', 'badamsoft-product-exporter-for-woocommerce' ),
                'description' => __( 'Generate export previews in the admin UI.', 'badamsoft-product-exporter-for-woocommerce' ),
            ],
            self::CAP_TEMPLATES         => [
                'label'       => __( 'Templates', 'badamsoft-product-exporter-for-woocommerce' ),
                'description' => __( 'Create, update, delete, import, export, and apply export templates.', 'badamsoft-product-exporter-for-woocommerce' ),
            ],
            self::CAP_HISTORY_VIEW      => [
                'label'       => __( 'Export history', 'badamsoft-product-exporter-for-woocommerce' ),
                'description' => __( 'View export run history and download files.', 'badamsoft-product-exporter-for-woocommerce' ),
            ],
            self::CAP_HISTORY_ACTIONS   => [
                'label'       => __( 'Export history actions', 'badamsoft-product-exporter-for-woocommerce' ),
                'description' => __( 'Delete export history runs and access bulk actions.', 'badamsoft-product-exporter-for-woocommerce' ),
            ],
            self::CAP_HISTORY_MANAGE    => [
                'label'       => __( 'Manage export history', 'badamsoft-product-exporter-for-woocommerce' ),
                'description' => __( 'Delete export history runs.', 'badamsoft-product-exporter-for-woocommerce' ),
            ],
            self::CAP_SETTINGS          => [
                'label'       => __( 'Settings', 'badamsoft-product-exporter-for-woocommerce' ),
                'description' => __( 'Access the Settings tab and modify plugin configuration.', 'badamsoft-product-exporter-for-woocommerce' ),
            ],
        ];

        $filtered = apply_filters( 'prodexfo_capabilities', $capabilities );

        return is_array( $filtered ) ? $filtered : $capabilities;
    }

    /**
     * @return array<string, bool>
     */
    public function get_capability_flags_for_user( ?int $user_id = null ): array {
        if ( null === $user_id ) {
            $user_id = get_current_user_id();
        }

        $is_admin = user_can( $user_id, 'manage_options' );

        $base_flags = [
            self::CAP_MANUAL_EXPORT   => $is_admin,
            self::CAP_PREVIEW_EXPORT  => $is_admin,
            self::CAP_TEMPLATES       => $is_admin,
            self::CAP_HISTORY_VIEW    => $is_admin,
            self::CAP_HISTORY_ACTIONS => $is_admin,
            self::CAP_HISTORY_MANAGE  => $is_admin,
            self::CAP_SETTINGS        => $is_admin,
        ];

        $filtered = apply_filters( 'prodexfo_capability_flags', $base_flags, $user_id );

        $flags = is_array( $filtered ) ? $filtered : $base_flags;

        // Never allow external filters to weaken or alter the default FREE capability flags.
        foreach ( $base_flags as $cap => $value ) {
            $flags[ $cap ] = $value;
        }

        foreach ( $flags as $cap => $value ) {
            if ( ! is_string( $cap ) || '' === $cap ) {
                unset( $flags[ $cap ] );
                continue;
            }

            $flags[ $cap ] = (bool) $value;
        }

        return $flags;
    }
}
