<?php

declare(strict_types=1);

/**
 * Admin page template for Badamsoft Product Exporter for WooCommerce.
 *
 * @var array<string, array<string, mixed>> $field_definitions
 * @var array<int, string>                  $default_selection
 * @var array<string, int>                  $filters
 * @var array<int, array<string, mixed>>    $category_options
 * @var array<int, array<string, mixed>>    $brand_options
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$prodexfo_group_labels = [
    'general'    => __( 'General', 'badamsoft-product-exporter-for-woocommerce' ),
    'pricing'    => __( 'Pricing', 'badamsoft-product-exporter-for-woocommerce' ),
    'inventory'  => __( 'Inventory', 'badamsoft-product-exporter-for-woocommerce' ),
    'content'    => __( 'Content', 'badamsoft-product-exporter-for-woocommerce' ),
    'taxonomy'   => __( 'Taxonomies', 'badamsoft-product-exporter-for-woocommerce' ),
    'media'      => __( 'Media', 'badamsoft-product-exporter-for-woocommerce' ),
    'attributes' => __( 'Attributes', 'badamsoft-product-exporter-for-woocommerce' ),
    'acf'        => __( 'ACF Fields', 'badamsoft-product-exporter-for-woocommerce' ),
    'meta'       => __( 'Meta Fields', 'badamsoft-product-exporter-for-woocommerce' ),
];

$prodexfo_fields_by_group = [];

foreach ( $field_definitions as $prodexfo_key => $prodexfo_definition ) {
    $prodexfo_group                                   = $prodexfo_definition['group'] ?? 'other';
    $prodexfo_fields_by_group[ $prodexfo_group ][ $prodexfo_key ] = $prodexfo_definition;
}

$prodexfo_selected_category = $filters['category'] ?? 0;
$prodexfo_selected_brand    = $filters['brand'] ?? 0;
$prodexfo_selected_tags     = $filters['exclude_tags'] ?? [];
$prodexfo_selected_categories_exclude = $filters['exclude_categories'] ?? [];

$prodexfo_format_number = static function ( $prodexfo_value ): string {
    if ( null === $prodexfo_value ) {
        return '';
    }

    if ( is_numeric( $prodexfo_value ) ) {
        $prodexfo_value = (string) $prodexfo_value;
    }

    return $prodexfo_value;
};

$prodexfo_regular_range  = $filters['price']['regular'] ?? [ 'min' => null, 'max' => null ];
$prodexfo_sale_range     = $filters['price']['sale'] ?? [ 'min' => null, 'max' => null ];
$prodexfo_stock_filters  = $filters['stock'] ?? [ 'min' => null, 'max' => null, 'only_in_stock' => false, 'only_zero' => false ];
$prodexfo_created_range  = $filters['date_created'] ?? [ 'from' => null, 'to' => null ];
$prodexfo_modified_range = $filters['date_modified'] ?? [ 'from' => null, 'to' => null ];
$prodexfo_condition_value = esc_attr( wp_json_encode( $filters['condition_groups'] ?? [] ) );
?>
<?php
$prodexfo_can_manage_templates = isset( $can_manage_templates ) ? (bool) $can_manage_templates : true;
$prodexfo_can_preview          = isset( $can_preview ) ? (bool) $can_preview : true;
?>
<div class="wrap wooproduct-exporter">
    <header class="wooproduct-exporter__headline">
        <div class="wooproduct-exporter__headline-copy">
            <h1><?php esc_html_e( 'Badamsoft Product Exporter for WooCommerce', 'badamsoft-product-exporter-for-woocommerce' ); ?></h1>
            <p class="description">
                <?php esc_html_e( 'Select the data columns and file format for the export.', 'badamsoft-product-exporter-for-woocommerce' ); ?>
            </p>
        </div>
        <div class="wooproduct-exporter__headline-meta">
            <div class="wooproduct-exporter__logo-slot">
                <img class="wooproduct-exporter__logo" src="<?php echo esc_url( $plugin_url . 'assets/logo.png' ); ?>" alt="<?php esc_attr_e( 'BadamSoft logo', 'badamsoft-product-exporter-for-woocommerce' ); ?>" width="180" height="52" />
            </div>
            <div class="wooproduct-exporter__actions">
                <button type="submit" form="wooproduct-exporter-form" class="button button-primary button-hero">
                    <?php esc_html_e( 'Export Products to CSV', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                </button>
            </div>
        </div>
    </header>

    <form id="wooproduct-exporter-form" method="post" class="wooproduct-exporter__form">
        <?php wp_nonce_field( 'prodexfo_manual_export_action', 'prodexfo_manual_export_nonce' ); ?>
        <input type="hidden" name="wc_product_csv_export" value="1" />

        <div class="wooproduct-exporter__toolbar">
            <button type="button" class="button button-secondary wooproduct-exporter__select-all">
                <?php esc_html_e( 'Select all', 'badamsoft-product-exporter-for-woocommerce' ); ?>
            </button>
            <div class="wooproduct-exporter__toolbar-spacer"></div>
            <?php if ( $prodexfo_can_preview ) : ?>
                <button type="button" class="button button-secondary" data-preview-trigger>
                    <?php esc_html_e( 'Preview first 20 rows', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                </button>
            <?php else : ?>
                <span class="description">
                    <?php esc_html_e( 'Preview is disabled for your role.', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                </span>
            <?php endif; ?>
        </div>

        <section class="wc-pce-selected-fields" data-field-order-panel>
            <header>
                <div>
                    <h2><?php esc_html_e( 'Selected columns', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>
                    <p><?php esc_html_e( 'Drag to reorder columns as they will appear in the export.', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                </div>
                <small data-field-order-hint hidden></small>
            </header>
            <div class="wc-pce-selected-fields__body">
                <ul class="wc-pce-selected-fields__list" data-field-order-list>
                    <li class="wc-pce-selected-fields__empty" data-field-order-empty>
                        <?php esc_html_e( 'No fields selected yet. Choose columns in the sections below.', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                    </li>
                </ul>
                <input type="hidden" name="fields_order" value="" data-field-order-input />
            </div>
        </section>

        <?php if ( $prodexfo_can_manage_templates ) : ?>
            <section class="wc-pce-template-panel" data-template-panel>
                <header>
                    <div>
                        <h2><?php esc_html_e( 'Export templates', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>
                        <p><?php esc_html_e( 'Save field selections, filters, and file settings for quick reuse.', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                    </div>
                    <div class="wc-pce-template-status" data-template-status hidden></div>
                </header>
                <div class="wc-pce-template-grid">
                    <label class="wc-pce-template-select">
                        <span><?php esc_html_e( 'Choose template', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                        <select class="wooproduct-exporter__select" data-template-select>
                            <option value=""><?php esc_html_e( '— No template —', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                            <?php foreach ( $template_options as $prodexfo_template ) : ?>
                                <option value="<?php echo esc_attr( $prodexfo_template['id'] ); ?>" <?php selected( $selected_template_id, $prodexfo_template['id'] ); ?>>
                                    <?php echo esc_html( $prodexfo_template['name'] ); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <small data-template-description></small>
                    </label>
                    <div class="wc-pce-template-actions">
                        <button type="button" class="button button-secondary" data-template-save><?php esc_html_e( 'Save as new', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                        <button type="button" class="button button-secondary" data-template-update><?php esc_html_e( 'Update current', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                        <button type="button" class="button" data-template-delete><?php esc_html_e( 'Delete', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                        <button type="button" class="button" data-template-export><?php esc_html_e( 'Export CSV', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                    </div>
                </div>
                <div class="wc-pce-template-import" data-template-import>
                    <div class="wc-pce-template-import__drop" data-template-dropzone>
                        <p><?php esc_html_e( 'Drag & drop CSV template here or', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                        <button type="button" class="button button-secondary" data-template-import-trigger><?php esc_html_e( 'Choose file', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                        <input type="file" accept=".csv,text/csv,application/vnd.ms-excel" data-template-import-input hidden />
                        <label class="wc-pce-template-import__replace">
                            <input type="checkbox" data-template-import-replace />
                            <span><?php esc_html_e( 'Replace templates with the same ID', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                        </label>
                    </div>
                </div>
            </section>
        <?php else : ?>
            <section class="wc-pce-template-panel">
                <header>
                    <div>
                        <h2><?php esc_html_e( 'Export templates', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>
                        <p><?php esc_html_e( 'Template management is not available for your role.', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                    </div>
                </header>
            </section>
        <?php endif; ?>

        <section class="wooproduct-exporter__filters">
            <header>
                <h2><?php esc_html_e( 'Filters', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>
                <p><?php esc_html_e( 'Refine the export dataset before choosing the fields.', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
            </header>
            <div class="wooproduct-exporter__filters-grid">
                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Category', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="filter_category" class="wooproduct-exporter__select">
                        <option value="0" <?php selected( 0, $prodexfo_selected_category ); ?>><?php esc_html_e( 'All categories', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <?php foreach ( $category_options as $prodexfo_option ) : ?>
                            <option value="<?php echo esc_attr( $prodexfo_option['id'] ); ?>" <?php selected( $prodexfo_option['id'], $prodexfo_selected_category ); ?>>
                                <?php echo esc_html( $prodexfo_option['name'] ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Brand', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="filter_brand" class="wooproduct-exporter__select" <?php disabled( empty( $brand_options ) ); ?>>
                        <option value="0" <?php selected( 0, $prodexfo_selected_brand ); ?>><?php esc_html_e( 'All brands', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <?php foreach ( $brand_options as $prodexfo_option ) : ?>
                            <option value="<?php echo esc_attr( $prodexfo_option['id'] ); ?>" <?php selected( $prodexfo_option['id'], $prodexfo_selected_brand ); ?>>
                                <?php echo esc_html( $prodexfo_option['name'] ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <?php if ( empty( $brand_options ) ) : ?>
                        <span class="wooproduct-exporter__filter-hint" data-brand-hint>
                            <?php esc_html_e( 'No brand taxonomy detected. Configure a brand attribute to enable this filter.', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                        </span>
                    <?php endif; ?>
                </label>
            </div>

            <div class="wooproduct-exporter__filters-grid wooproduct-exporter__filters-grid--stretch">
                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Regular price range', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <div class="wooproduct-exporter__range">
                        <input type="number" step="0.01" name="filter_regular_price_min" placeholder="<?php esc_attr_e( 'Min', 'badamsoft-product-exporter-for-woocommerce' ); ?>" value="<?php echo esc_attr( $prodexfo_format_number( $prodexfo_regular_range['min'] ?? null ) ); ?>" />
                        <span class="wooproduct-exporter__range-divider">—</span>
                        <input type="number" step="0.01" name="filter_regular_price_max" placeholder="<?php esc_attr_e( 'Max', 'badamsoft-product-exporter-for-woocommerce' ); ?>" value="<?php echo esc_attr( $prodexfo_format_number( $prodexfo_regular_range['max'] ?? null ) ); ?>" />
                    </div>
                </label>

                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Sale price range', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <div class="wooproduct-exporter__range">
                        <input type="number" step="0.01" name="filter_sale_price_min" placeholder="<?php esc_attr_e( 'Min', 'badamsoft-product-exporter-for-woocommerce' ); ?>" value="<?php echo esc_attr( $prodexfo_format_number( $prodexfo_sale_range['min'] ?? null ) ); ?>" />
                        <span class="wooproduct-exporter__range-divider">—</span>
                        <input type="number" step="0.01" name="filter_sale_price_max" placeholder="<?php esc_attr_e( 'Max', 'badamsoft-product-exporter-for-woocommerce' ); ?>" value="<?php echo esc_attr( $prodexfo_format_number( $prodexfo_sale_range['max'] ?? null ) ); ?>" />
                    </div>
                </label>

                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Discounted products', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="filter_discount_mode" class="wooproduct-exporter__select">
                        <option value="" <?php selected( '', $filters['discount_mode'] ?? '' ); ?>><?php esc_html_e( 'Any state', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <option value="discounted" <?php selected( 'discounted', $filters['discount_mode'] ?? '' ); ?>><?php esc_html_e( 'Only discounted', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <option value="non_discounted" <?php selected( 'non_discounted', $filters['discount_mode'] ?? '' ); ?>><?php esc_html_e( 'Without discount', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                    </select>
                </label>
            </div>

            <div class="wooproduct-exporter__filters-grid wooproduct-exporter__filters-grid--stretch">
                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Stock quantity range', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <div class="wooproduct-exporter__range">
                        <input type="number" step="1" name="filter_stock_min" placeholder="<?php esc_attr_e( 'Min', 'badamsoft-product-exporter-for-woocommerce' ); ?>" value="<?php echo esc_attr( $prodexfo_format_number( $prodexfo_stock_filters['min'] ?? null ) ); ?>" />
                        <span class="wooproduct-exporter__range-divider">—</span>
                        <input type="number" step="1" name="filter_stock_max" placeholder="<?php esc_attr_e( 'Max', 'badamsoft-product-exporter-for-woocommerce' ); ?>" value="<?php echo esc_attr( $prodexfo_format_number( $prodexfo_stock_filters['max'] ?? null ) ); ?>" />
                    </div>
                    <div class="wooproduct-exporter__checkbox-row">
                        <label><input type="checkbox" name="filter_stock_only_in_stock" value="1" <?php checked( ! empty( $prodexfo_stock_filters['only_in_stock'] ) ); ?> /> <?php esc_html_e( 'Only products in stock', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                        <label><input type="checkbox" name="filter_stock_only_zero" value="1" <?php checked( ! empty( $prodexfo_stock_filters['only_zero'] ) ); ?> /> <?php esc_html_e( 'Only zero stock', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                    </div>
                </label>

                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Created between', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <div class="wooproduct-exporter__range">
                        <input type="date" name="filter_created_from" value="<?php echo esc_attr( (string) ( $prodexfo_created_range['from'] ?? '' ) ); ?>" />
                        <span class="wooproduct-exporter__range-divider">→</span>
                        <input type="date" name="filter_created_to" value="<?php echo esc_attr( (string) ( $prodexfo_created_range['to'] ?? '' ) ); ?>" />
                    </div>
                </label>

                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Last modified between', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <div class="wooproduct-exporter__range">
                        <input type="date" name="filter_modified_from" value="<?php echo esc_attr( (string) ( $prodexfo_modified_range['from'] ?? '' ) ); ?>" />
                        <span class="wooproduct-exporter__range-divider">→</span>
                        <input type="date" name="filter_modified_to" value="<?php echo esc_attr( (string) ( $prodexfo_modified_range['to'] ?? '' ) ); ?>" />
                    </div>
                </label>
            </div>

            <div class="wooproduct-exporter__filters-grid wooproduct-exporter__filters-grid--stretch">
                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Image availability', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="filter_image_mode" class="wooproduct-exporter__select">
                        <option value="" <?php selected( '', $filters['image_mode'] ?? '' ); ?>><?php esc_html_e( 'Any state', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <option value="with_image" <?php selected( 'with_image', $filters['image_mode'] ?? '' ); ?>><?php esc_html_e( 'Only with featured image', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <option value="without_image" <?php selected( 'without_image', $filters['image_mode'] ?? '' ); ?>><?php esc_html_e( 'Only without featured image', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                    </select>
                </label>

                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Reviews', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="filter_reviews_mode" class="wooproduct-exporter__select">
                        <option value="" <?php selected( '', $filters['reviews_mode'] ?? '' ); ?>><?php esc_html_e( 'Any state', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <option value="with_reviews" <?php selected( 'with_reviews', $filters['reviews_mode'] ?? '' ); ?>><?php esc_html_e( 'Only products with reviews', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <option value="without_reviews" <?php selected( 'without_reviews', $filters['reviews_mode'] ?? '' ); ?>><?php esc_html_e( 'Only products without reviews', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                    </select>
                </label>

                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Search in descriptions', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <input type="search" name="filter_description_search" placeholder="<?php esc_attr_e( 'Keyword or phrase', 'badamsoft-product-exporter-for-woocommerce' ); ?>" value="<?php echo esc_attr( $filters['description_search'] ?? '' ); ?>" />
                    <small class="wooproduct-exporter__filter-hint"><?php esc_html_e( 'Matches both full and short description text.', 'badamsoft-product-exporter-for-woocommerce' ); ?></small>
                </label>
            </div>

            <div class="wooproduct-exporter__filters-grid wooproduct-exporter__filters-grid--stretch">
                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Exclude categories', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="filter_exclude_categories[]" class="wooproduct-exporter__select" multiple size="6">
                        <?php foreach ( $category_options as $prodexfo_option ) :
                            $prodexfo_is_selected = in_array( $prodexfo_option['id'], $prodexfo_selected_categories_exclude, true );
                            ?>
                            <option value="<?php echo esc_attr( $prodexfo_option['id'] ); ?>" <?php selected( true, $prodexfo_is_selected ); ?>>
                                <?php echo esc_html( $prodexfo_option['name'] ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <small class="wooproduct-exporter__filter-hint"><?php esc_html_e( 'Hold Ctrl / Cmd to pick multiple terms.', 'badamsoft-product-exporter-for-woocommerce' ); ?></small>
                </label>

                <label class="wooproduct-exporter__filter">
                    <span><?php esc_html_e( 'Exclude tags', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="filter_exclude_tags[]" class="wooproduct-exporter__select" multiple size="6">
                        <?php foreach ( $tag_options as $prodexfo_option ) :
                            $prodexfo_is_selected = in_array( $prodexfo_option['id'], $prodexfo_selected_tags, true );
                            ?>
                            <option value="<?php echo esc_attr( $prodexfo_option['id'] ); ?>" <?php selected( true, $prodexfo_is_selected ); ?>>
                                <?php echo esc_html( $prodexfo_option['name'] ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <small class="wooproduct-exporter__filter-hint"><?php esc_html_e( 'Filtering supports standard product tags.', 'badamsoft-product-exporter-for-woocommerce' ); ?></small>
                </label>
            </div>

            <section class="wooproduct-exporter__filters-advanced">
                <header>
                    <h3><?php esc_html_e( 'Conditional rule groups', 'badamsoft-product-exporter-for-woocommerce' ); ?></h3>
                    <p><?php esc_html_e( 'Combine multiple numeric conditions (price, stock, sales, reviews) using AND / OR logic for advanced scenarios.', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                </header>
                <div class="wc-pce-condition-builder" data-condition-builder>
                    <input type="hidden" name="filter_condition_groups" data-condition-builder-input value="<?php echo esc_attr( $prodexfo_condition_value ); ?>" />
                    <div class="wc-pce-condition-builder__body" data-condition-builder-body>
                        <p class="wc-pce-condition-builder__empty"><?php esc_html_e( 'Loading condition builder…', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                    </div>
                </div>
            </section>
        </section>

        <div class="wooproduct-exporter__field-groups">
            <?php foreach ( $prodexfo_fields_by_group as $prodexfo_group => $prodexfo_fields ) :
                $prodexfo_group_label = $prodexfo_group_labels[ $prodexfo_group ] ?? ucwords( str_replace( '_', ' ', $prodexfo_group ) );
                ?>
                <section class="wooproduct-exporter__group" data-group="<?php echo esc_attr( $prodexfo_group ); ?>">
                    <header>
                        <h2><?php echo esc_html( $prodexfo_group_label ); ?></h2>
                        <button type="button" class="button button-secondary wooproduct-exporter__toggle-group" data-target="<?php echo esc_attr( $prodexfo_group ); ?>">
                            <?php esc_html_e( 'Toggle group', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                        </button>
                    </header>
                    <div class="wooproduct-exporter__group-body">
                        <?php foreach ( $prodexfo_fields as $prodexfo_key => $prodexfo_field ) :
                            $prodexfo_is_checked = in_array( $prodexfo_key, $default_selection, true );
                            $prodexfo_is_private = ! empty( $prodexfo_field['is_private'] );
                            ?>
                            <label class="wooproduct-exporter__field" for="field-<?php echo esc_attr( $prodexfo_key ); ?>">
                                <span class="wooproduct-exporter__field-label">
                                    <?php echo esc_html( $prodexfo_field['label'] ?? $prodexfo_key ); ?>
                                    <?php if ( $prodexfo_is_private ) : ?>
                                        <span class="wooproduct-exporter__field-tag">
                                            <?php esc_html_e( 'Private', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                                        </span>
                                    <?php endif; ?>
                                </span>
                                <span class="wooproduct-exporter__field-control">
                                    <input
                                        id="field-<?php echo esc_attr( $prodexfo_key ); ?>"
                                        type="checkbox"
                                        name="fields[]"
                                        value="<?php echo esc_attr( $prodexfo_key ); ?>"
                                        <?php checked( $prodexfo_is_checked ); ?>
                                    />
                                </span>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </section>
            <?php endforeach; ?>
        </div>

        <div class="wooproduct-exporter__submit">
            <div class="wooproduct-exporter__format">
                <label>
                    <span><?php esc_html_e( 'Export format', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="export_format" class="wooproduct-exporter__select">
                        <?php foreach ( $format_options as $prodexfo_format_key => $prodexfo_format_label ) : ?>
                            <option value="<?php echo esc_attr( $prodexfo_format_key ); ?>" <?php selected( $prodexfo_format_key, $selected_format ); ?>>
                                <?php echo esc_html( $prodexfo_format_label ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>
            </div>
            <div class="wooproduct-exporter__file-settings">
                <label>
                    <span><?php esc_html_e( 'Delimiter', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="export_delimiter" class="wooproduct-exporter__select">
                        <?php foreach ( $delimiter_options as $prodexfo_value => $prodexfo_label ) : ?>
                            <option value="<?php echo esc_attr( $prodexfo_value ); ?>" <?php selected( $file_settings['delimiter'] ?? ',', $prodexfo_value ); ?>>
                                <?php echo esc_html( $prodexfo_label ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label>
                    <span><?php esc_html_e( 'Encoding', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <select name="export_encoding" class="wooproduct-exporter__select">
                        <?php foreach ( $encoding_options as $prodexfo_value => $prodexfo_label ) : ?>
                            <option value="<?php echo esc_attr( $prodexfo_value ); ?>" <?php selected( $file_settings['encoding'] ?? 'UTF-8', $prodexfo_value ); ?>>
                                <?php echo esc_html( $prodexfo_label ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label>
                    <span><?php esc_html_e( 'File name', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                    <input type="text" name="export_filename" value="<?php echo esc_attr( $file_settings['filename'] ?? 'wc-products-export-{{date}}' ); ?>" />
                    <small><?php esc_html_e( 'Use {{date}} to insert the current timestamp.', 'badamsoft-product-exporter-for-woocommerce' ); ?></small>
                </label>
                <label class="wooproduct-exporter__checkbox">
                    <input type="checkbox" name="export_attach_images_zip" value="1" <?php checked( ! empty( $file_settings['attach_images_zip'] ) ); ?> />
                    <span><?php esc_html_e( 'Attach ZIP archive with product images to the export result.', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                </label>
            </div>
            <button type="submit" class="button button-primary button-hero">
                <?php esc_html_e( 'Export Products', 'badamsoft-product-exporter-for-woocommerce' ); ?>
            </button>
        </div>
    </form>

    <div class="wc-pce-modal" data-template-modal hidden>
        <div class="wc-pce-modal__dialog">
            <form data-template-modal-form>
                <header class="wc-pce-modal__header">
                    <h2 data-template-modal-title><?php esc_html_e( 'Save template', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>
                    <button type="button" class="button button-link" data-template-modal-close aria-label="<?php esc_attr_e( 'Close modal', 'badamsoft-product-exporter-for-woocommerce' ); ?>">×</button>
                </header>

                <div class="wc-pce-modal__body">
                    <div class="wc-pce-form-row">
                        <label for="wc-pce-template-modal-name"><?php esc_html_e( 'Template name', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                        <input type="text" id="wc-pce-template-modal-name" name="template_name" required />
                    </div>

                    <div class="wc-pce-form-row">
                        <label for="wc-pce-template-modal-description"><?php esc_html_e( 'Description (optional)', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                        <textarea id="wc-pce-template-modal-description" name="template_description" rows="3"></textarea>
                    </div>

                    <div class="wc-pce-form-row wc-pce-form-row--inline">
                        <label class="wc-pce-form-checkbox">
                            <input type="checkbox" name="template_attach_images_zip" data-template-attach-images-zip />
                            <span><?php esc_html_e( 'Attach ZIP archive of product images when using this template.', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                        </label>
                    </div>
                </div>

                <footer class="wc-pce-modal__footer">
                    <button type="button" class="button" data-template-modal-cancel><?php esc_html_e( 'Cancel', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                    <button type="submit" class="button button-primary" data-template-modal-submit><?php esc_html_e( 'Save template', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                </footer>
            </form>
        </div>
    </div>

    <div class="wc-pce-modal wc-pce-modal--preview" data-preview-modal hidden>
        <div class="wc-pce-modal__dialog">
            <header class="wc-pce-modal__header">
                <h2><?php esc_html_e( 'Preview (first 20 rows)', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>
                <button type="button" class="button button-link" data-preview-close aria-label="<?php esc_attr_e( 'Close preview', 'badamsoft-product-exporter-for-woocommerce' ); ?>">×</button>
            </header>
            <div class="wc-pce-modal__body">
                <div class="wc-pce-preview__loading" data-preview-loading hidden>
                    <span class="spinner is-active"></span>
                    <p><?php esc_html_e( 'Generating preview…', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                </div>
                <div class="wc-pce-preview__error" data-preview-error hidden></div>
                <p class="wc-pce-preview__summary" data-preview-summary hidden></p>
                <div class="wc-pce-preview__table" data-preview-table-wrapper>
                    <table class="widefat fixed" data-preview-table>
                        <thead></thead>
                        <tbody></tbody>
                    </table>
                </div>
            </div>
            <footer class="wc-pce-modal__footer">
                <button type="button" class="button" data-preview-close><?php esc_html_e( 'Close', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
            </footer>
        </div>
    </div>
</div>
