<?php

declare(strict_types=1);

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$prodexfo_weekday_buttons = [
    1 => __( 'Mon', 'badamsoft-product-exporter-for-woocommerce' ),
    2 => __( 'Tue', 'badamsoft-product-exporter-for-woocommerce' ),
    3 => __( 'Wed', 'badamsoft-product-exporter-for-woocommerce' ),
    4 => __( 'Thu', 'badamsoft-product-exporter-for-woocommerce' ),
    5 => __( 'Fri', 'badamsoft-product-exporter-for-woocommerce' ),
    6 => __( 'Sat', 'badamsoft-product-exporter-for-woocommerce' ),
    0 => __( 'Sun', 'badamsoft-product-exporter-for-woocommerce' ),
];

$prodexfo_date_format_option    = get_option( 'date_format', 'Y-m-d' );
$prodexfo_time_format_option    = get_option( 'time_format', 'H:i' );
$prodexfo_schedule_datetime_fmt = trim( $prodexfo_date_format_option . ' ' . $prodexfo_time_format_option );

$prodexfo_format_task_datetime = static function ( ?string $prodexfo_value, \DateTimeZone $prodexfo_timezone, string $prodexfo_format ): string {
    if ( empty( $prodexfo_value ) ) {
        return '';
    }

    $prodexfo_date = \DateTimeImmutable::createFromFormat( 'Y-m-d H:i:s', $prodexfo_value, new \DateTimeZone( 'UTC' ) );

    if ( ! $prodexfo_date ) {
        return '';
    }

    return $prodexfo_date->setTimezone( $prodexfo_timezone )->format( $prodexfo_format );
};
?>
<div class="wrap wc-pce-scheduled-exports">
    <h1><?php esc_html_e( 'Scheduled Exports', 'badamsoft-product-exporter-for-woocommerce' ); ?></h1>
    <p class="description"><?php esc_html_e( 'Manage automated export tasks. Create schedules, adjust incremental settings, and trigger runs on demand.', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>

    <?php settings_errors( 'prodexfo_schedules' ); ?>

    <p>
        <button type="button" class="button button-primary" data-schedule-modal-open><?php esc_html_e( 'Create schedule', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
    </p>

    <h2><?php esc_html_e( 'Existing schedules', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>

    <table class="widefat fixed striped">
        <thead>
            <tr>
                <th><?php esc_html_e( 'Name', 'badamsoft-product-exporter-for-woocommerce' ); ?></th>
                <th><?php esc_html_e( 'Template', 'badamsoft-product-exporter-for-woocommerce' ); ?></th>
                <th><?php esc_html_e( 'Schedule', 'badamsoft-product-exporter-for-woocommerce' ); ?></th>
                <th><?php esc_html_e( 'Status', 'badamsoft-product-exporter-for-woocommerce' ); ?></th>
                <th><?php esc_html_e( 'Last run', 'badamsoft-product-exporter-for-woocommerce' ); ?></th>
                <th><?php esc_html_e( 'Next run', 'badamsoft-product-exporter-for-woocommerce' ); ?></th>
                <th><?php esc_html_e( 'Actions', 'badamsoft-product-exporter-for-woocommerce' ); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php if ( empty( $tasks ) ) : ?>
                <tr>
                    <td colspan="7"><?php esc_html_e( 'No scheduled exports yet.', 'badamsoft-product-exporter-for-woocommerce' ); ?></td>
                </tr>
            <?php else : ?>
                <?php foreach ( $tasks as $prodexfo_task ) :
                    $prodexfo_template_name    = $templates[ $prodexfo_task['template_id'] ]['name'] ?? $prodexfo_task['template_id'];
                    $prodexfo_schedule_payload = $prodexfo_task['schedule_payload'] ?? [];

                    if ( is_string( $prodexfo_schedule_payload ) ) {
                        $prodexfo_decoded = json_decode( $prodexfo_schedule_payload, true );
                        $prodexfo_schedule_payload = is_array( $prodexfo_decoded ) ? $prodexfo_decoded : [];
                    }

                    $prodexfo_task_timezone = \BadamSoft\WooProductExporter\Helpers\Utils::create_timezone( ! empty( $prodexfo_task['schedule_timezone'] ) ? (string) $prodexfo_task['schedule_timezone'] : null );

                    $prodexfo_last_run_display = $prodexfo_format_task_datetime( $prodexfo_task['last_run_at'] ?? null, $prodexfo_task_timezone, $prodexfo_schedule_datetime_fmt );
                    $prodexfo_next_run_display = $prodexfo_format_task_datetime( $prodexfo_task['next_run_at'] ?? null, $prodexfo_task_timezone, $prodexfo_schedule_datetime_fmt );

                    $prodexfo_task_actions = [];

                    if ( isset( $prodexfo_task['actions'] ) && is_array( $prodexfo_task['actions'] ) ) {
                        $prodexfo_task_actions = $prodexfo_task['actions'];
                    }

                    $prodexfo_task_edit_payload = [
                        'id'                => isset( $prodexfo_task['id'] ) ? (int) $prodexfo_task['id'] : 0,
                        'name'              => $prodexfo_task['name'] ?? '',
                        'template_id'       => $prodexfo_task['template_id'] ?? '',
                        'incremental'       => isset( $prodexfo_task['incremental'] ) ? (int) $prodexfo_task['incremental'] : 0,
                        'incremental_field' => $prodexfo_task['incremental_field'] ?? 'post_modified',
                        'schedule_type'     => $prodexfo_task['schedule_type'] ?? 'weekly',
                        'schedule_payload'  => $prodexfo_schedule_payload,
                        'schedule_interval' => isset( $prodexfo_task['schedule_interval'] ) ? (int) $prodexfo_task['schedule_interval'] : 0,
                        'schedule_cron'     => $prodexfo_task['schedule_cron'] ?? '',
                        'schedule_timezone' => $prodexfo_task['schedule_timezone'] ?? ( \BadamSoft\WooProductExporter\Helpers\Utils::create_timezone()->getName() ),
                        'enabled'           => isset( $prodexfo_task['enabled'] ) ? (int) $prodexfo_task['enabled'] : 1,
                        'actions'           => $prodexfo_task_actions,
                    ];

                    $prodexfo_task_edit_payload_json = wp_json_encode( $prodexfo_task_edit_payload );

                    if ( false === $prodexfo_task_edit_payload_json ) {
                        $prodexfo_task_edit_payload_json = '';
                    }
                    ?>
                    <tr>
                        <td><?php echo esc_html( $prodexfo_task['name'] ); ?></td>
                        <td><?php echo esc_html( $prodexfo_template_name ); ?></td>
                        <td>
                            <?php
                            switch ( $prodexfo_task['schedule_type'] ) {
                                case 'interval':
                                    $prodexfo_minutes = $prodexfo_task['schedule_interval'] ? ( $prodexfo_task['schedule_interval'] / MINUTE_IN_SECONDS ) : 0;
                                    /* translators: %d: interval in minutes */
                                    echo esc_html( sprintf( __( 'Every %d minutes', 'badamsoft-product-exporter-for-woocommerce' ), max( 1, (int) $prodexfo_minutes ) ) );
                                    break;
                                case 'weekly':
                                    $prodexfo_payload       = is_array( $prodexfo_schedule_payload ) ? $prodexfo_schedule_payload : [];
                                    $prodexfo_days          = isset( $prodexfo_payload['days'] ) && is_array( $prodexfo_payload['days'] ) ? array_map( 'intval', $prodexfo_payload['days'] ) : [];
                                    $prodexfo_times         = isset( $prodexfo_payload['times'] ) && is_array( $prodexfo_payload['times'] ) ? array_map( 'sanitize_text_field', $prodexfo_payload['times'] ) : [];
                                    $prodexfo_times_by_day  = isset( $prodexfo_payload['times_by_day'] ) && is_array( $prodexfo_payload['times_by_day'] ) ? $prodexfo_payload['times_by_day'] : [];
                                    $prodexfo_mode          = isset( $prodexfo_payload['mode'] ) ? sanitize_key( $prodexfo_payload['mode'] ) : 'shared';
                                    $prodexfo_week_days_map = $prodexfo_weekday_buttons;

                                    $prodexfo_mode = in_array( $prodexfo_mode, [ 'shared', 'per_day' ], true ) ? $prodexfo_mode : 'shared';
                                    $prodexfo_times = array_values( array_filter( $prodexfo_times ) );

                                    if ( 'per_day' === $prodexfo_mode && ! empty( $prodexfo_times_by_day ) ) {
                                        echo '<ul class="wc-pce-schedule-list">';

                                        foreach ( $prodexfo_times_by_day as $prodexfo_day_key => $prodexfo_day_times ) {
                                            $prodexfo_day_index = is_numeric( $prodexfo_day_key ) ? (int) $prodexfo_day_key : -1;

                                            if ( ! isset( $prodexfo_week_days_map[ $prodexfo_day_index ] ) ) {
                                                continue;
                                            }

                                            $prodexfo_sanitized_times = is_array( $prodexfo_day_times ) ? array_values( array_filter( array_map( 'sanitize_text_field', $prodexfo_day_times ) ) ) : [];

                                            if ( empty( $prodexfo_sanitized_times ) ) {
                                                continue;
                                            }

                                            printf(
                                                '<li>%1$s: %2$s</li>',
                                                esc_html( $prodexfo_week_days_map[ $prodexfo_day_index ] ),
                                                esc_html( implode( ', ', $prodexfo_sanitized_times ) )
                                            );
                                        }

                                        echo '</ul>';
                                    } else {
                                        $prodexfo_day_names = [];

                                        foreach ( $prodexfo_days as $prodexfo_day_index ) {
                                            if ( isset( $prodexfo_week_days_map[ $prodexfo_day_index ] ) ) {
                                                $prodexfo_day_names[] = $prodexfo_week_days_map[ $prodexfo_day_index ];
                                            }
                                        }

                                        $prodexfo_time_label = empty( $prodexfo_times ) ? '—' : ( count( $prodexfo_times ) === 1 ? $prodexfo_times[0] : implode( ', ', $prodexfo_times ) );

                                        printf(
                                            /* translators: 1: comma-separated day names, 2: time */
                                            esc_html__( 'Weekly on %1$s at %2$s', 'badamsoft-product-exporter-for-woocommerce' ),
                                            esc_html( implode( ', ', $prodexfo_day_names ) ),
                                            esc_html( $prodexfo_time_label )
                                        );
                                    }
                                    break;
                                case 'monthly':
                                    $prodexfo_payload = is_array( $prodexfo_schedule_payload ) ? $prodexfo_schedule_payload : [];
                                    $prodexfo_days    = isset( $prodexfo_payload['days'] ) && is_array( $prodexfo_payload['days'] ) ? array_map( 'intval', $prodexfo_payload['days'] ) : [];
                                    $prodexfo_times   = isset( $prodexfo_payload['times'] ) && is_array( $prodexfo_payload['times'] ) ? array_map( 'sanitize_text_field', $prodexfo_payload['times'] ) : [];

                                    $prodexfo_days  = array_values( array_filter( array_unique( $prodexfo_days ) ) );
                                    $prodexfo_times = array_values( array_filter( $prodexfo_times ) );

                                    if ( empty( $prodexfo_days ) || empty( $prodexfo_times ) ) {
                                        esc_html_e( 'Monthly schedule', 'badamsoft-product-exporter-for-woocommerce' );
                                        break;
                                    }

                                    sort( $prodexfo_days );

                                    $prodexfo_ordinal_days = array_map(
                                        static function ( int $prodexfo_day ): string {
                                            $prodexfo_day = max( 1, min( 31, $prodexfo_day ) );
                                            $prodexfo_timestamp = mktime( 0, 0, 0, 1, $prodexfo_day );

                                            return date_i18n( 'jS', $prodexfo_timestamp );
                                        },
                                        $prodexfo_days
                                    );

                                    if ( count( $prodexfo_ordinal_days ) === 1 ) {
                                        /* translators: %s: day of month (ordinal) */
                                        $prodexfo_day_label = sprintf( __( 'the %s', 'badamsoft-product-exporter-for-woocommerce' ), $prodexfo_ordinal_days[0] );
                                    } else {
                                        /* translators: %s: comma-separated list of month days (ordinals) */
                                        $prodexfo_day_label = sprintf( __( 'the %s', 'badamsoft-product-exporter-for-woocommerce' ), implode( ', ', $prodexfo_ordinal_days ) );
                                    }

                                    $prodexfo_time_label = count( $prodexfo_times ) === 1 ? $prodexfo_times[0] : implode( ', ', $prodexfo_times );

                                    printf(
                                        /* translators: 1: day(s) of month (ordinal), 2: time */
                                        esc_html__( 'Monthly on %1$s at %2$s', 'badamsoft-product-exporter-for-woocommerce' ),
                                        esc_html( $prodexfo_day_label ),
                                        esc_html( $prodexfo_time_label )
                                    );
                                    break;
                                case 'cron':
                                    $prodexfo_cron_expression = isset( $prodexfo_task['schedule_cron'] ) ? trim( (string) $prodexfo_task['schedule_cron'] ) : '';

                                    if ( $prodexfo_cron_expression ) {
                                        /* translators: %s: cron expression */
                                        printf( esc_html__( 'Cron: %s', 'badamsoft-product-exporter-for-woocommerce' ), esc_html( $prodexfo_cron_expression ) );
                                    } else {
                                        esc_html_e( 'Custom schedule', 'badamsoft-product-exporter-for-woocommerce' );
                                    }
                                    break;
                                default:
                                    echo esc_html( __( 'Custom schedule', 'badamsoft-product-exporter-for-woocommerce' ) );
                                    break;
                            }
                            ?>
                            <?php if ( $task_timezone_string ) : ?>
                                <div class="description">
                                    <?php
                                    /* translators: %s: timezone name */
                                    printf( esc_html__( 'Timezone: %s', 'badamsoft-product-exporter-for-woocommerce' ), esc_html( $task_timezone_string ) );
                                    ?>
                                </div>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ( empty( $prodexfo_task['enabled'] ) ) : ?>
                                <span class="status status-disabled"><?php esc_html_e( 'Disabled', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                            <?php elseif ( 'running' === $prodexfo_task['status'] ) : ?>
                                <span class="status status-running"><?php esc_html_e( 'Running', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                            <?php elseif ( 'error' === $prodexfo_task['status'] ) : ?>
                                <span class="status status-error"><?php esc_html_e( 'Error', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                            <?php else : ?>
                                <span class="status status-idle"><?php esc_html_e( 'Idle', 'badamsoft-product-exporter-for-woocommerce' ); ?></span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ( $prodexfo_last_run_display ) : ?>
                                <?php echo esc_html( $prodexfo_last_run_display ); ?>
                            <?php else : ?>
                                —
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ( $prodexfo_next_run_display ) : ?>
                                <?php echo esc_html( $prodexfo_next_run_display ); ?>
                                <span class="description"><?php echo esc_html( $task_timezone_string ); ?></span>
                            <?php else : ?>
                                —
                            <?php endif; ?>
                        </td>
                        <td>
                            <div class="wc-pce-task-actions">
                                <button type="button" class="button button-small" data-schedule-edit='<?php echo esc_attr( $prodexfo_task_edit_payload_json ? $prodexfo_task_edit_payload_json : '' ); ?>'><?php esc_html_e( 'Edit', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                                <form method="post" class="wc-pce-task-actions__form">
                                    <?php wp_nonce_field( 'prodexfo_schedule_action', 'prodexfo_schedule_nonce' ); ?>
                                    <input type="hidden" name="prodexfo_schedule_action" value="toggle" />
                                    <input type="hidden" name="task_id" value="<?php echo isset( $prodexfo_task['id'] ) ? (int) $prodexfo_task['id'] : 0; ?>" />
                                    <button type="submit" name="prodexfo_schedule_action" value="toggle" class="button button-small"><?php echo empty( $prodexfo_task['enabled'] ) ? esc_html__( 'Enable', 'badamsoft-product-exporter-for-woocommerce' ) : esc_html__( 'Disable', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                                    <button type="submit" name="prodexfo_schedule_action" value="delete" class="button button-small" onclick="return confirm('<?php echo esc_js( __( 'Delete this schedule?', 'badamsoft-product-exporter-for-woocommerce' ) ); ?>');"><?php esc_html_e( 'Delete', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                                </form>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<div class="wc-pce-modal" data-schedule-modal data-default-timezone="<?php echo esc_attr( $timezone ); ?>" data-default-weekday="1" data-default-time="09:00" data-default-monthly-day="1" hidden>
    <div class="wc-pce-modal__dialog">
        <header class="wc-pce-modal__header">
            <h2 data-schedule-modal-title data-title-create="<?php echo esc_attr_x( 'Scheduling options', 'modal create title', 'badamsoft-product-exporter-for-woocommerce' ); ?>" data-title-edit="<?php echo esc_attr_x( 'Edit schedule', 'modal edit title', 'badamsoft-product-exporter-for-woocommerce' ); ?>"><?php esc_html_e( 'Scheduling options', 'badamsoft-product-exporter-for-woocommerce' ); ?></h2>
            <button type="button" class="button button-link" data-schedule-modal-close aria-label="<?php esc_attr_e( 'Close', 'badamsoft-product-exporter-for-woocommerce' ); ?>">×</button>
        </header>
        <form method="post" data-schedule-form>
            <?php wp_nonce_field( 'prodexfo_schedule_action', 'prodexfo_schedule_nonce' ); ?>
            <input type="hidden" name="prodexfo_schedule_action" value="create" />
            <input type="hidden" name="task_schedule_payload" data-schedule-payload />
            <input type="hidden" name="task_schedule_type" data-schedule-type />
            <input type="hidden" name="task_schedule_interval" data-schedule-interval value="0" />
            <input type="hidden" name="task_enabled" data-schedule-enabled value="1" />
            <input type="hidden" name="task_id" value="" data-schedule-task-id />
            <div class="wc-pce-modal__body">
                <div class="wc-pce-form-row">
                    <label for="schedule_task_name"><?php esc_html_e( 'Schedule name', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                    <input type="text" id="schedule_task_name" name="task_name" required />
                </div>
                <div class="wc-pce-form-row">
                    <label for="schedule_task_template"><?php esc_html_e( 'Template', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                    <select id="schedule_task_template" name="task_template" required>
                        <option value=""><?php esc_html_e( 'Select template', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        <?php foreach ( $templates as $prodexfo_template ) : ?>
                            <option value="<?php echo esc_attr( $prodexfo_template['id'] ); ?>"><?php echo esc_html( $prodexfo_template['name'] ?? $prodexfo_template['id'] ); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <fieldset class="wc-pce-form-row wc-pce-schedule-mode" data-schedule-mode>
                    <legend><?php esc_html_e( 'Scheduling mode', 'badamsoft-product-exporter-for-woocommerce' ); ?></legend>
                    <label><input type="radio" name="schedule_mode" value="none" /> <?php esc_html_e( 'Do not schedule', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                    <label><input type="radio" name="schedule_mode" value="automatic" checked /> <?php esc_html_e( 'Automatic scheduling', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                    <label><input type="radio" name="schedule_mode" value="manual" /> <?php esc_html_e( 'Manual (cron)', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                </fieldset>

                <div class="wc-pce-schedule-panel" data-schedule-panel="automatic" hidden>
                    <fieldset class="wc-pce-form-row" data-automatic-type>
                        <legend><?php esc_html_e( 'Automatic schedule', 'badamsoft-product-exporter-for-woocommerce' ); ?></legend>
                        <label><input type="radio" name="automatic_type" value="weekly" checked /> <?php esc_html_e( 'Every selected week days', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                        <label><input type="radio" name="automatic_type" value="monthly" /> <?php esc_html_e( 'Every month on date', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                    </fieldset>

                    <div class="wc-pce-automatic-field" data-automatic="weekly">
                        <p><?php esc_html_e( 'Run on days:', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                            <div class="wc-pce-day-picker" data-weekly-days data-weekday-labels='<?php echo esc_attr( wp_json_encode( array_map( 'strval', $prodexfo_weekday_buttons ) ) ); ?>'>
                            <?php foreach ( $prodexfo_weekday_buttons as $prodexfo_index => $prodexfo_label ) : ?>
                                <button type="button" class="button" data-day="<?php echo (int) $prodexfo_index; ?>"><?php echo esc_html( $prodexfo_label ); ?></button>
                            <?php endforeach; ?>
                        </div>
                        <div class="wc-pce-weekly-mode" data-weekly-mode>
                            <p class="description" data-weekly-mode-label data-label-shared="<?php echo esc_attr__( 'Using the same time slots for the selected days.', 'badamsoft-product-exporter-for-woocommerce' ); ?>" data-label-per-day="<?php /* translators: %s: weekday label */ echo esc_attr__( 'Custom time slots for %s.', 'badamsoft-product-exporter-for-woocommerce' ); ?>" data-label-none="<?php echo esc_attr__( 'Select at least one day to configure time slots.', 'badamsoft-product-exporter-for-woocommerce' ); ?>"></p>
                        </div>

                        <div class="wc-pce-weekly-times-group" data-weekly-shared>
                            <p><?php esc_html_e( 'Times of day:', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                            <div class="wc-pce-time-chips" data-weekly-times></div>
                            <div class="wc-pce-time-adder">
                                <input type="time" data-weekly-time-input value="09:00" />
                                <button type="button" class="button button-secondary" data-weekly-time-add><?php esc_html_e( 'Add time', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                            </div>
                        </div>

                        <div class="wc-pce-weekly-times-group" data-weekly-custom hidden>
                            <p><?php esc_html_e( 'Times per day:', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                            <div class="wc-pce-weekly-custom" data-weekly-custom-container>
                                <p class="description" data-weekly-custom-empty><?php esc_html_e( 'Select a day above to add time slots.', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                                <div class="wc-pce-weekly-custom-display" data-weekly-custom-display hidden>
                                    <div class="wc-pce-weekly-custom-header">
                                        <strong data-weekly-custom-heading></strong>
                                        <button type="button" class="button-link" data-weekly-custom-remove-day><?php esc_html_e( 'Remove day from schedule', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                                    </div>
                                    <div class="wc-pce-time-chips" data-weekly-custom-times></div>
                                    <div class="wc-pce-time-adder">
                                        <input type="time" data-weekly-custom-time-input value="09:00" />
                                        <button type="button" class="button button-secondary" data-weekly-custom-add><?php esc_html_e( 'Add time', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="wc-pce-automatic-field" data-automatic="monthly" hidden>
                        <p><?php esc_html_e( 'Run on day of month:', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                        <input type="number" min="1" max="31" value="1" data-monthly-day />
                        <p><?php esc_html_e( 'Times of day:', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                        <div class="wc-pce-time-chips" data-monthly-times></div>
                        <div class="wc-pce-time-adder">
                            <input type="time" data-monthly-time-input value="09:00" />
                            <button type="button" class="button button-secondary" data-monthly-time-add><?php esc_html_e( 'Add time', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                        </div>
                    </div>

                    <div class="wc-pce-form-row">
                        <label for="schedule_timezone"><?php esc_html_e( 'Timezone', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                        <select id="schedule_timezone" name="task_timezone">
                            <?php
                            echo \BadamSoft\WooProductExporter\Helpers\TimezoneSelect::render( $timezone ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                            ?>
                        </select>
                    </div>
                </div>

                <div class="wc-pce-schedule-panel" data-schedule-panel="manual" hidden>
                    <label for="schedule_cron_expression"><?php esc_html_e( 'Cron expression', 'badamsoft-product-exporter-for-woocommerce' ); ?></label>
                    <input type="text" id="schedule_cron_expression" name="task_cron_expression" placeholder="0 3 * * *" />
                    <p class="description"><?php esc_html_e( 'Use standard cron format (minute hour day month weekday).', 'badamsoft-product-exporter-for-woocommerce' ); ?></p>
                </div>

                <fieldset class="wc-pce-form-row">
                    <legend><?php esc_html_e( 'Incremental export', 'badamsoft-product-exporter-for-woocommerce' ); ?></legend>
                    <label>
                        <input type="checkbox" name="task_incremental" value="1" />
                        <?php esc_html_e( 'Export only products changed since last run', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                    </label>
                    <label for="schedule_incremental_field">
                        <?php esc_html_e( 'Field', 'badamsoft-product-exporter-for-woocommerce' ); ?>
                        <select id="schedule_incremental_field" name="task_incremental_field">
                            <option value="post_modified"><?php esc_html_e( 'Date modified', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                            <option value="post_date"><?php esc_html_e( 'Date created', 'badamsoft-product-exporter-for-woocommerce' ); ?></option>
                        </select>
                    </label>
                </fieldset>
            </div>
            <footer class="wc-pce-modal__footer">
                <button type="button" class="button" data-schedule-modal-close><?php esc_html_e( 'Cancel', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
                <button type="submit" class="button button-primary" data-schedule-submit data-label-create="<?php echo esc_attr_x( 'Save schedule', 'modal create submit', 'badamsoft-product-exporter-for-woocommerce' ); ?>" data-label-edit="<?php echo esc_attr_x( 'Update schedule', 'modal edit submit', 'badamsoft-product-exporter-for-woocommerce' ); ?>"><?php esc_html_e( 'Save schedule', 'badamsoft-product-exporter-for-woocommerce' ); ?></button>
            </footer>
        </form>
    </div>
</div>
